<?php
/**
 * Plugin Name: LearnDash LMS - Integrity
 * Plugin URI: https://www.learndash.com/
 * Description: Protect your LearnDash site from content theft.
 * Version: 1.2.1
 * Author: LearnDash
 * Author URI: https://www.learndash.com/
 * Text Domain: learndash-integrity
 * Domain Path: languages
 *
 * @package LearnDash\Integrity
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once plugin_dir_path( __FILE__ ) . 'vendor/autoload.php';
require_once plugin_dir_path( __FILE__ ) . 'vendor-prefixed/autoload.php';

use LearnDash\Core\Autoloader;
use LearnDash\Integrity\Dependency_Checker;
use LearnDash\Integrity\Plugin;
use LearnDash\Integrity\StellarWP\Assets\Config;

define( 'LEARNDASH_INTEGRITY_VERSION', '1.2.1' );
define( 'LEARNDASH_INTEGRITY_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );
define( 'LEARNDASH_INTEGRITY_PLUGIN_URL', plugins_url( '/', __FILE__ ) );
define( 'LEARNDASH_INTEGRITY_PLUGIN_FILE', __FILE__ );

add_action(
	'plugins_loaded',
	function () {
		load_plugin_textdomain(
			'learndash-integrity',
			false,
			plugin_basename(
				__DIR__
			) . '/languages'
		);
	}
);

$learndash_integrity_dependency_checker = new Dependency_Checker();

$learndash_integrity_dependency_checker->set_dependencies(
	[
		'sfwd-lms/sfwd_lms.php' => [
			'label'            => '<a href="https://www.learndash.com" target="_blank">' . __( 'LearnDash LMS', 'learndash-integrity' ) . '</a>',
			'class'            => 'SFWD_LMS',
			'version_constant' => 'LEARNDASH_VERSION',
			'min_version'      => '4.6.0',
		],
	]
);

$learndash_integrity_dependency_checker->set_message(
	esc_html__( 'LearnDash LMS - Integrity requires the following plugin(s) to be active:', 'learndash-integrity' )
);

add_action(
	'learndash_init',
	function () use ( $learndash_integrity_dependency_checker ) {
		// If plugin requirements aren't met, don't run anything else to prevent possible fatal errors.
		if ( ! $learndash_integrity_dependency_checker->check_dependency_results() || php_sapi_name() === 'cli' ) {
			return;
		}

		// Sets up configuration for StellarWP Assets library.
		Config::set_hook_prefix( 'learndash_integrity' );
		Config::set_path( LEARNDASH_INTEGRITY_PLUGIN_PATH );
		Config::set_version( LEARNDASH_INTEGRITY_VERSION );
		Config::set_relative_asset_path( 'assets/' );

		learndash_integrity_extra_includes();
		learndash_integrity_extra_autoloading();

		learndash_register_provider( Plugin::class );
	}
);

/**
 * Includes extra files.
 *
 * @since 1.2.1
 *
 * @return void
 */
function learndash_integrity_extra_includes(): void {
	if ( is_admin() ) {
		include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/admin/class-settings-page.php';
		include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/admin/class-settings-section.php';
	}

	include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/class-utilities.php';
	include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/class-prevent-hotlinking.php';
	include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/class-prevent-concurrent-login.php';
	include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/class-prevent-content-copy.php';
	include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/class-recaptcha.php';
	include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/class-recaptcha-v3.php';
	include LEARNDASH_INTEGRITY_PLUGIN_PATH . 'includes/class-recaptcha-v2.php';
}

/**
 * Set ups autoloader for extra classes and files.
 *
 * @since 1.2.1
 *
 * @return void
 */
function learndash_integrity_extra_autoloading(): void {
	if ( ! class_exists( Autoloader::class ) ) {
		return;
	}

	$autoloader = Autoloader::instance();

	// Iterate through all files under ./src/deprecated.
	$iterator = new RecursiveDirectoryIterator( LEARNDASH_INTEGRITY_PLUGIN_PATH . 'src/deprecated/' );
	$files    = new RecursiveIteratorIterator( $iterator, RecursiveIteratorIterator::SELF_FIRST );

	foreach ( $files as $file ) {
		if (
			! $file instanceof SplFileInfo
			|| ! $file->isFile()
			|| $file->getExtension() !== 'php'
		) {
			continue;
		}

		if ( strstr( $file->getRealPath(), 'functions' ) ) {
			// If this was named functions.php in any directory, load it.
			include_once $file->getRealPath();
		} else {
			// Construct the proper Class Name based on the file path.
			$class_name = str_replace(
				'/',
				'\\',
				(string) preg_replace(
					'/.*?src\/deprecated\/(.*?)\.php/',
					'$1',
					$file->getRealPath()
				)
			);

			$autoloader->register_class( $class_name, $file->getRealPath() );
		}
	}

	$autoloader->register_autoloader();
}
