<?php
/**
 * Base Recaptcha class file.
 *
 * @package LearnDash\Integrity
 */

namespace LearnDash\Integrity;

use WP_Error;
use WP_Post;
use WP_User;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Base ReCaptcha class.
 *
 * @since 1.0.0
 */
abstract class ReCaptcha {
	/**
	 * Nonce action.
	 *
	 * @since 1.2.1
	 *
	 * @var string
	 */
	protected const NONCE_ACTION = 'learndash_integrity_verify_recaptcha';

	/**
	 * Nonce field key.
	 *
	 * @since 1.2.1
	 *
	 * @var string
	 */
	protected const NONCE_KEY = 'learndash_integrity_nonce';

	/**
	 * Settings.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	protected $settings;

	/**
	 * Location of the recaptcha.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	protected $location;

	/**
	 * Setting key.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $setting_key;

	/**
	 * Token name.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $token_name;

	/**
	 * Cache the last result from recaptcha, for preventing many calls to registration_errors.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	protected $last_result;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->settings = get_option( 'learndash_settings_ld_integrity' );
		$this->location = $this->settings['location'] ?? [];
	}

	/**
	 * Register scripts.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	abstract public function register_scripts();

	/**
	 * Verifies the captcha token.
	 *
	 * @since 1.0.0
	 *
	 * @param string $token A hash that is returned by Google recaptcha.
	 *
	 * @return mixed
	 */
	abstract protected function verify_captcha( string $token );

	/**
	 * Filters POST token data to always return string.
	 *
	 * @since 1.2.0
	 *
	 * @return string
	 */
	private function filter_token(): string {
		if (
			! isset( $_POST[ self::NONCE_KEY ] )
			|| ! wp_verify_nonce(
				sanitize_text_field( wp_unslash( $_POST[ self::NONCE_KEY ] ) ),
				self::NONCE_ACTION
			)
		) {
			wp_die( 'Invalid nonce.' );
		}

		$token = isset( $_POST[ $this->token_name ] )
			? sanitize_text_field( wp_unslash( $_POST[ $this->token_name ] ) )
			: '';

		if ( ! is_string( $token ) ) {
			$token = '';
		}

		return $token;
	}

	/**
	 * Registers the hooks.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	protected function add_hooks() {
		add_action( 'wp_loaded', [ $this, 'register_scripts' ] );

		if ( in_array( 'login', $this->location, true ) ) {
			// Do the verification when login.
			add_filter( 'wp_authenticate_user', [ $this, 'verify_captcha_login' ], 9 );
		}

		if ( in_array( 'register', $this->location, true ) ) {
			if ( ! is_multisite() ) {
				add_filter( 'registration_errors', [ $this, 'verify_captcha_register' ], 9 );
			} else {
				add_action( 'signup_extra_fields', [ $this, 'display_signup_recaptcha' ] );
				add_action( 'signup_blogform', [ $this, 'display_signup_recaptcha' ] );
				add_filter( 'wpmu_validate_user_signup', [ $this, 'verify_captcha_register_multi' ], 10 );
			}

			add_filter( 'learndash_registration_errors', [ $this, 'register_captcha_error_ld_registration' ] );
		}
	}

	/**
	 * Checks if the recaptcha scripts should be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function should_load_recaptcha_scripts(): bool {
		if (
			doing_action( 'after_signup_form' )
			|| doing_action( 'before_signup_form' )
		) {
			return is_multisite();
		}

		if (
			doing_action( 'wp_head' )
			|| doing_action( 'wp_footer' )
		) {
			$id = get_the_ID();

			if ( $id === false ) {
				return false;
			}

			$post = get_post( $id );

			return ! is_admin()
				&& ! is_user_logged_in()
				&& $post instanceof WP_Post
				&& (
					'sfwd-courses' === $post->post_type
					|| has_shortcode( $post->post_content, 'learndash_login' )
					|| has_block( 'learndash/ld-login', $post->post_content )
					|| has_block( 'learndash/ld-registration', $post->post_content )
					|| $this->is_login_shortcode_found_in_menu()
				);
		}

		return true;
	}

	/**
	 * Returns true if the login shortcode is found in a menu item description.
	 *
	 * @since 1.2.0
	 *
	 * @return bool
	 */
	public function is_login_shortcode_found_in_menu(): bool {
		$login_shortcode_found_in_menu = false;

		foreach ( get_nav_menu_locations() as $term_id ) {
			if ( $login_shortcode_found_in_menu ) {
				break;
			}

			$menu_items = wp_get_nav_menu_items( $term_id );

			if ( is_array( $menu_items ) ) {
				foreach ( $menu_items as $menu_item ) {
					if ( has_shortcode( $menu_item->description, 'learndash_login' ) ) {
						$login_shortcode_found_in_menu = true;
						break;
					}
				}
			}
		}

		return $login_shortcode_found_in_menu;
	}

	/**
	 * Verifies captcha in multisite registration.
	 *
	 * @since 1.0.0
	 *
	 * @param array $result The result of the registration.
	 *
	 * @phpstan-param array{
	 *     user_name: string,
	 *     user_email: string,
	 *     orig_username: string,
	 *     errors?: WP_Error,
	 * } $result
	 *
	 * @return mixed
	 */
	public function verify_captcha_register_multi( $result ) {
		global $current_user;

		if ( is_admin() && ! defined( 'DOING_AJAX' ) && ! empty( $current_user->data->ID ) ) {
			return $result;
		}

		if ( ! empty( $result['errors'] ) ) {
			$errors = $result['errors'];
		} else {
			$errors = new \WP_Error();
		}

		$token = $this->filter_token();

		if ( empty( $token ) ) {
			$errors->add(
				'captcha_error',
				__( '<strong>Error:</strong> Please complete the captcha', 'learndash-integrity' )
			);

			$result['errors'] = $errors;
		} elseif ( ! $this->verify_captcha( $token ) ) {
			$errors->add(
				'captcha_error',
				__( 'Invalid captcha.', 'learndash-integrity' )
			);

			$result['errors'] = $errors;
		}

		return $result;
	}


	/**
	 * Displays signup recaptcha error element.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_Error|null $errors WP_Error object or null.
	 *
	 * @return void
	 */
	public function display_signup_recaptcha( $errors ) {
		if ( is_wp_error( $errors ) ) {
			$error_message = $errors->get_error_message( 'captcha_error' );
			if ( ! empty( $error_message ) ) {
				printf( '<p class="error">%s</p>', esc_html( $error_message ) );
			}
		}
	}

	/**
	 * Verifies captcha on registration.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_Error $errors Existing WP_Error object.
	 *
	 * @return WP_Error
	 */
	public function verify_captcha_register( $errors ) {
		if ( defined( 'XMLRPC_REQUEST' ) ) {
			return $errors;
		}

		$token = $this->filter_token();

		if ( empty( $token ) ) {
			$errors->add(
				'captcha_error',
				__( 'Invalid captcha.', 'learndash-integrity' )
			);
		} elseif ( false === $this->verify_captcha( $token ) ) {
			$errors->add(
				'captcha_error',
				__( 'Invalid captcha.', 'learndash-integrity' )
			);
		}

		return $errors;
	}

	/**
	 * Verifies captcha on login.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_User|WP_Error $user User object or WP_Error.
	 *
	 * @return WP_User|WP_Error
	 */
	public function verify_captcha_login( $user ) {
		if ( defined( 'XMLRPC_REQUEST' ) ) {
			return $user;
		}

		if (
			isset( $_SERVER['REQUEST_METHOD'] )
			&& $_SERVER['REQUEST_METHOD'] === 'POST'
		) {
			$token = $this->filter_token();

			if ( empty( $token ) ) {
				if ( is_wp_error( $user ) ) {
					$user->add(
						'captcha_error',
						__( '<strong>Error:</strong> Please complete the captcha', 'learndash-integrity' )
					);
				} else {
					return new \WP_Error(
						'captcha_error',
						__( '<strong>Error:</strong> Please complete the captcha', 'learndash-integrity' )
					);
				}
			} elseif ( false === $this->verify_captcha( $token ) ) {
				if ( is_wp_error( $user ) ) {
					$user->add(
						'captcha_error',
						__( '<strong>Error:</strong> Invalid captcha.', 'learndash-integrity' )
					);
				} else {
					return new \WP_Error(
						'captcha_error',
						__( '<strong>Error:</strong> Invalid captcha.', 'learndash-integrity' )
					);
				}
			}
		}

		return $user;
	}

	/**
	 * Filters LD registration errors messages.
	 *
	 * @since 1.0.0
	 *
	 * @param array<string, string> $errors Error messages.
	 *
	 * @return array<string, string>
	 */
	public function register_captcha_error_ld_registration( $errors ) {
		$errors['captcha_error'] = __( 'Please complete the captcha', 'learndash-integrity' );

		return $errors;
	}

	/**
	 * Check if the current class implementation is enabled.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function is_enabled(): bool {
		if (
			! isset( $this->settings['recaptcha'] )
			|| 'yes' !== $this->settings['recaptcha']
		) {
			return false;
		}

		if (
			! isset( $this->settings[ $this->setting_key ] )
			|| 'yes' !== $this->settings[ $this->setting_key ]
		) {
			return false;
		}

		if ( ! count( $this->location ) ) {
			return false;
		}

		return true;
	}
}
