<?php
/**
 * ReCaptcha V3 class file.
 *
 * @package LearnDash\Integrity
 */

namespace LearnDash\Integrity;

use LearnDash\Integrity\StellarWP\Assets\Asset;
use LearnDash\Integrity\StellarWP\Assets\Assets;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * ReCaptcha V3 class.
 *
 * @since 1.0.0
 */
class reCaptcha_V3 extends ReCaptcha { // phpcs:ignore PEAR.NamingConventions.ValidClassName.StartWithCapital, Generic.Classes.OpeningBraceSameLine.ContentAfterBrace -- Keeping the class name for backward compatibility.
	/**
	 * Handle name for the reCaptcha script.
	 *
	 * @since 1.2.1
	 *
	 * @var string
	 */
	private const RECAPTCHA_SDK = 'learndash-integrity-recaptcha-v3-sdk';

	/**
	 * Handle name for the reCaptcha v3 implementation script.
	 *
	 * @since 1.2.1
	 *
	 * @var string
	 */
	private const RECAPTCHA = 'learndash-integrity-recaptcha-v3';

	/**
	 * Setting key for the reCaptcha v3.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $setting_key = 'v3';

	/**
	 * Token name for the reCaptcha v3.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $token_name = 'token';

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct();

		if ( true === $this->is_enabled() ) {
			$this->add_hooks();

			add_filter( 'script_loader_src', [ $this, 'filter_asset_src' ], 10, 2 );
		}
	}

	/**
	 * Gets the element selectors for recaptcha v3.
	 *
	 * @since 1.2.1
	 *
	 * @return string
	 */
	private function get_element_selectors(): string {
		$el = [];

		if ( in_array( 'login', $this->location, true ) ) {
			$el[] = '#loginform';
		}

		if ( in_array( 'register', $this->location, true ) ) {
			$el[] = '#registerform, #setupform, #learndash_registerform';
		}

		return implode( ',', $el );
	}

	/**
	 * Verifies captcha token.
	 *
	 * @since 1.0.0
	 *
	 * @param string $token A hash that is returned by Google recaptcha.
	 *
	 * @return mixed
	 */
	protected function verify_captcha( string $token ) {
		if ( ! empty( $this->last_result ) ) {
			if (
				false === $this->last_result['success']
				|| $this->settings['score_threshold'] > $this->last_result['score']
			) {
				return false;
			}

			return true;
		}

		$url      = 'https://www.google.com/recaptcha/api/siteverify';
		$data     = [
			'secret'   => $this->settings['secret_key'],
			'response' => $token,
		];
		$response = wp_remote_post(
			$url,
			[
				'body' => $data,
			]
		);
		$body     = wp_remote_retrieve_body( $response );
		$body     = json_decode( $body, true );

		$this->last_result = $body;

		if (
			false === $body['success']
			|| $this->settings['score_threshold'] > $body['score']
		) {
			return false;
		}

		return true;
	}

	/**
	 * Register scripts used in this class.
	 *
	 * @since 1.0.0
	 * @since 1.2.1 Updated to use StellarWP Assets library.
	 *
	 * @return void
	 */
	public function register_scripts() {
		Asset::add(
			self::RECAPTCHA_SDK,
			'https://www.google.com/recaptcha/api.js'
		)
			->enqueue_on( 'login_head' )
			->enqueue_on( 'before_signup_form' )
			->enqueue_on( 'learndash_registration_form_fields_before' )
			->enqueue_on( 'wp_head' )
			->set_condition( [ $this, 'should_load_recaptcha_scripts' ] )
			->register();

		$el = $this->get_element_selectors();

		Asset::add(
			self::RECAPTCHA,
			'js/recaptcha-v3.js'
		)
			->add_localize_script(
				'learndash.integrity.recaptcha.v3',
				[
					'elements'   => $el,
					'siteKey'    => $this->settings['site_key'],
					'nonceKey'   => static::NONCE_KEY,
					'nonceValue' => wp_create_nonce( static::NONCE_ACTION ),
				]
			)
			->enqueue_on( 'login_footer' )
			->enqueue_on( 'after_signup_form' )
			->enqueue_on( 'learndash_registration_form_after' )
			->enqueue_on( 'wp_footer' )
			->set_dependencies( 'jquery' )
			->set_condition( [ $this, 'should_load_recaptcha_scripts' ] )
			->in_footer()
			->register();
	}

	/**
	 * Filters asset URL.
	 *
	 * @since 1.2.1
	 *
	 * @param string $src    URL of the script.
	 * @param string $handle Handle of the script.
	 *
	 * @return string
	 */
	public function filter_asset_src( $src, $handle ) {
		if ( $handle !== self::RECAPTCHA_SDK ) {
			return $src;
		}

		// Set query args to match the exact query args that the official Google recaptcha script recommends.
		return add_query_arg(
			[
				'render' => $this->settings['site_key'],
				'ver'    => null, // Unset the version query arg added by WP.
			],
			$src
		);
	}
}

new reCaptcha_V3();
