<?php
/**
 * ReCaptcha v2 implementation class file.
 *
 * @package LearnDash\Integrity
 */

namespace LearnDash\Integrity;

use LearnDash\Integrity\StellarWP\Assets\Asset;
use LearnDash\Integrity\StellarWP\Assets\Assets;
use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * ReCaptcha V2 class.
 *
 * @since 1.0.0
 */
class reCaptcha_V2 extends ReCaptcha { // phpcs:ignore PEAR.NamingConventions.ValidClassName.StartWithCapital, Generic.Classes.OpeningBraceSameLine.ContentAfterBrace -- Keeping the class name for backward compatibility.
	/**
	 * Handle for the reCaptcha script.
	 *
	 * @since 1.2.1
	 *
	 * @var string
	 */
	private const RECAPTCHA_SDK = 'learndash-integrity-recaptcha-v2-sdk';

	/**
	 * Handle for the reCaptcha v2 implementation script.
	 *
	 * @since 1.2.1
	 *
	 * @var string
	 */
	private const RECAPTCHA = 'learndash-integrity-recaptcha-v2';

	/**
	 * Setting key for the reCaptcha v2.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $setting_key = 'v2';

	/**
	 * Token name for the reCaptcha v2.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $token_name = 'g-recaptcha-response';

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct();

		if ( true === $this->is_enabled() ) {
			$this->add_hooks();

			add_filter( 'script_loader_src', [ $this, 'filter_asset_src' ], 10, 2 );

			if ( in_array( 'login', $this->location, true ) ) {
				add_action( 'login_form', [ $this, 'show_captcha_checkbox' ] );
				add_filter( 'login_form_middle', [ $this, 'show_captcha_checkbox_inline' ] );
			}

			if ( in_array( 'register', $this->location, true ) ) {
				if ( ! is_multisite() ) {
					add_action( 'register_form', [ $this, 'show_captcha_checkbox' ] );
				}
			}
		}
	}

	/**
	 * Displays signup recaptcha element.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_Error|null $errors WP_Error object.
	 *
	 * @return void
	 */
	public function display_signup_recaptcha( $errors ) {
		if ( is_wp_error( $errors ) ) {
			$error_message = $errors->get_error_message( 'captcha_error' );

			if ( ! empty( $error_message ) ) {
				printf( '<p class="error">%s</p>', esc_html( $error_message ) );
			}
		}

		$this->show_captcha_checkbox();
	}

	/**
	 * Verifies captcha token.
	 *
	 * @since 1.0.0
	 *
	 * @param string $token A hash that is returned by Google recaptcha.
	 *
	 * @return mixed
	 */
	protected function verify_captcha( string $token ) {
		if ( ! empty( $this->last_result ) ) {
			return $this->last_result['success'];
		}

		$url               = 'https://www.google.com/recaptcha/api/siteverify';
		$data              = [
			'secret'   => $this->settings['secret_key_v2'],
			'response' => $token,
		];
		$response          = wp_remote_post(
			$url,
			[
				'body' => $data,
			]
		);
		$body              = wp_remote_retrieve_body( $response );
		$body              = json_decode( $body, true );
		$this->last_result = $body;

		return $body['success'];
	}

	/**
	 * Returns captcha checkbox inline wrapper.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content Content.
	 *
	 * @return string
	 */
	public function show_captcha_checkbox_inline( $content ) {
		$content .= '<div class="ld_captcha_el"></div>';
		$content .= wp_nonce_field( static::NONCE_ACTION, static::NONCE_KEY, true, false );

		return $content;
	}

	/**
	 * Shows captcha checkbox wrapper.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function show_captcha_checkbox() {
		echo '<div class="ld_captcha_el"></div>';
		wp_nonce_field( static::NONCE_ACTION, static::NONCE_KEY );
	}

	/**
	 * Register scripts used in this class.
	 *
	 * @since 1.0.0
	 * @since 1.2.1 Updated to use StellarWP Assets library.
	 *
	 * @return void
	 */
	public function register_scripts() {
		Asset::add(
			self::RECAPTCHA_SDK,
			'https://www.google.com/recaptcha/api.js'
		)
			->set_as_async()
			->set_as_deferred()
			->enqueue_on( 'login_head' )
			->enqueue_on( 'before_signup_form' )
			->enqueue_on( 'learndash_registration_form_fields_before' )
			->enqueue_on( 'wp_head' )
			->set_condition( [ $this, 'should_load_recaptcha_scripts' ] )
			->register();

		Asset::add(
			self::RECAPTCHA,
			'js/recaptcha-v2.js'
		)
			->add_localize_script(
				'learndash.integrity.recaptcha.v2',
				[
					'siteKey' => $this->settings['site_key_v2'],
				]
			)
			->enqueue_on( 'login_footer' )
			->enqueue_on( 'after_signup_form' )
			->enqueue_on( 'learndash_registration_form_after' )
			->enqueue_on( 'wp_footer' )
			->set_condition( [ $this, 'should_load_recaptcha_scripts' ] )
			->in_footer()
			->register();
	}

	/**
	 * Filters asset URL.
	 *
	 * @since 1.2.1
	 *
	 * @param string $src    URL of the script.
	 * @param string $handle Handle of the script.
	 *
	 * @return string
	 */
	public function filter_asset_src( $src, $handle ) {
		if ( $handle !== self::RECAPTCHA_SDK ) {
			return $src;
		}

		// Set query args to match the exact query args that the official Google recaptcha script recommends.
		return add_query_arg(
			[
				'onload' => 'ldIntegrityRecaptchaV2OnloadCallback', // User defined callback function.
				'render' => 'explicit',
				'ver'    => null, // Unset the version query arg added by WP.
			],
			$src
		);
	}
}

new reCaptcha_V2();
